'use client'

import { useState, useEffect } from 'react'
import { useParams, useRouter } from 'next/navigation'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Textarea } from '@/components/ui/textarea'
import { Badge } from '@/components/ui/badge'
import { 
  Car, 
  Calendar,
  Users,
  Fuel,
  Settings,
  MapPin,
  ArrowLeft,
  Check,
  Clock,
  Shield
} from 'lucide-react'
import Link from 'next/link'

export default function CarDetailPage() {
  const params = useParams()
  const router = useRouter()
  const [car, setCar] = useState(null)
  const [loading, setLoading] = useState(true)
  const [bookingForm, setBookingForm] = useState({
    startDate: '',
    endDate: '',
    pickupLocation: '',
    notes: '',
    name: '',
    email: '',
    phone: ''
  })
  const [isBooking, setIsBooking] = useState(false)
  const [bookingSuccess, setBookingSuccess] = useState(false)

  useEffect(() => {
    if (params.id) {
      fetchCar()
    }
  }, [params.id])

  const fetchCar = async () => {
    try {
      const response = await fetch(`/api/cars/${params.id}`)
      const data = await response.json()
      if (data.success) {
        setCar(data.data)
      }
    } catch (error) {
      console.error('Failed to fetch car:', error)
    } finally {
      setLoading(false)
    }
  }

  const calculateTotalPrice = () => {
    if (!bookingForm.startDate || !bookingForm.endDate || !car) return 0
    
    const start = new Date(bookingForm.startDate)
    const end = new Date(bookingForm.endDate)
    const days = Math.ceil((end.getTime() - start.getTime()) / (1000 * 60 * 60 * 24)) + 1
    return days * car.price
  }

  const handleBooking = async (e) => {
    e.preventDefault()
    setIsBooking(true)

    try {
      // First create or get user
      const userResponse = await fetch('/api/users', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          name: bookingForm.name,
          email: bookingForm.email,
          phone: bookingForm.phone
        })
      })

      const userData = await userResponse.json()
      
      if (!userData.success) {
        throw new Error('Failed to create user')
      }

      // Create booking
      const bookingResponse = await fetch('/api/bookings', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          userId: userData.data.id,
          carId: car.id,
          startDate: bookingForm.startDate,
          endDate: bookingForm.endDate,
          pickupLocation: bookingForm.pickupLocation,
          notes: bookingForm.notes
        })
      })

      const bookingData = await bookingResponse.json()

      if (bookingData.success) {
        setBookingSuccess(true)
        // Reset form
        setBookingForm({
          startDate: '',
          endDate: '',
          pickupLocation: '',
          notes: '',
          name: '',
          email: '',
          phone: ''
        })
      } else {
        alert(bookingData.error || 'Booking failed')
      }
    } catch (error) {
      console.error('Booking error:', error)
      alert('Booking failed. Please try again.')
    } finally {
      setIsBooking(false)
    }
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading...</p>
        </div>
      </div>
    )
  }

  if (!car) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <Car className="h-16 w-16 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-semibold text-gray-900 mb-2">
            Mobil tidak ditemukan
          </h3>
          <Link href="/cars">
            <Button>Kembali ke Katalog</Button>
          </Link>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <Link href="/" className="flex items-center space-x-2">
              <Car className="h-8 w-8 text-blue-600" />
              <span className="text-xl font-bold text-gray-900">Rental Mobil</span>
            </Link>
            <Link href="/cars">
              <Button variant="outline">
                <ArrowLeft className="mr-2 h-4 w-4" />
                Kembali ke Katalog
              </Button>
            </Link>
          </div>
        </div>
      </header>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="grid lg:grid-cols-3 gap-8">
          {/* Car Details */}
          <div className="lg:col-span-2">
            <Card>
              <CardContent className="p-0">
                <div className="aspect-video bg-gray-100 relative">
                  {car.image ? (
                    <img 
                      src={car.image} 
                      alt={car.name}
                      className="w-full h-full object-cover"
                    />
                  ) : (
                    <div className="w-full h-full flex items-center justify-center">
                      <Car className="h-24 w-24 text-gray-400" />
                    </div>
                  )}
                  <Badge className="absolute top-4 right-4 bg-blue-600 text-lg px-3 py-1">
                    Rp {car.price.toLocaleString('id-ID')}/hari
                  </Badge>
                  {car.available && (
                    <Badge className="absolute top-4 left-4 bg-green-600">
                      Tersedia
                    </Badge>
                  )}
                </div>
                
                <div className="p-6">
                  <h1 className="text-3xl font-bold text-gray-900 mb-2">{car.name}</h1>
                  <p className="text-xl text-gray-600 mb-6">{car.brand} {car.model}</p>
                  
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                    <div className="flex items-center text-gray-600">
                      <Calendar className="h-5 w-5 mr-2 text-blue-600" />
                      <div>
                        <p className="text-sm">Tahun</p>
                        <p className="font-semibold">{car.year}</p>
                      </div>
                    </div>
                    <div className="flex items-center text-gray-600">
                      <Users className="h-5 w-5 mr-2 text-blue-600" />
                      <div>
                        <p className="text-sm">Kapasitas</p>
                        <p className="font-semibold">{car.seats} Seats</p>
                      </div>
                    </div>
                    <div className="flex items-center text-gray-600">
                      <Settings className="h-5 w-5 mr-2 text-blue-600" />
                      <div>
                        <p className="text-sm">Transmisi</p>
                        <p className="font-semibold">
                          {car.transmission === 'MANUAL' ? 'Manual' : 'Automatic'}
                        </p>
                      </div>
                    </div>
                    <div className="flex items-center text-gray-600">
                      <Fuel className="h-5 w-5 mr-2 text-blue-600" />
                      <div>
                        <p className="text-sm">Bahan Bakar</p>
                        <p className="font-semibold">
                          {car.fuelType === 'GASOLINE' ? 'Bensin' : 
                           car.fuelType === 'DIESEL' ? 'Diesel' :
                           car.fuelType === 'HYBRID' ? 'Hybrid' : 'Electric'}
                        </p>
                      </div>
                    </div>
                  </div>

                  {car.description && (
                    <div className="mb-6">
                      <h3 className="text-lg font-semibold mb-2">Deskripsi</h3>
                      <p className="text-gray-600">{car.description}</p>
                    </div>
                  )}

                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div className="bg-blue-50 rounded-lg p-4 text-center">
                      <Shield className="h-8 w-8 text-blue-600 mx-auto mb-2" />
                      <h4 className="font-semibold">Asuransi</h4>
                      <p className="text-sm text-gray-600">Perlindungan lengkap</p>
                    </div>
                    <div className="bg-green-50 rounded-lg p-4 text-center">
                      <Clock className="h-8 w-8 text-green-600 mx-auto mb-2" />
                      <h4 className="font-semibold">24/7 Support</h4>
                      <p className="text-sm text-gray-600">Bantuan kapan saja</p>
                    </div>
                    <div className="bg-purple-50 rounded-lg p-4 text-center">
                      <Check className="h-8 w-8 text-purple-600 mx-auto mb-2" />
                      <h4 className="font-semibold">Kualitas Terjamin</h4>
                      <p className="text-sm text-gray-600">Mobil terawat prima</p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Booking Form */}
          <div className="lg:col-span-1">
            <Card className="sticky top-4">
              <CardHeader>
                <CardTitle>Form Pemesanan</CardTitle>
              </CardHeader>
              <CardContent>
                {bookingSuccess ? (
                  <div className="text-center py-6">
                    <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                      <Check className="h-8 w-8 text-green-600" />
                    </div>
                    <h3 className="text-lg font-semibold text-gray-900 mb-2">
                      Pemesanan Berhasil!
                    </h3>
                    <p className="text-gray-600 mb-4">
                      Pesanan Anda telah diterima dan akan segera diproses.
                    </p>
                    <Button onClick={() => setBookingSuccess(false)} className="w-full">
                      Buat Pesanan Baru
                    </Button>
                  </div>
                ) : (
                  <form onSubmit={handleBooking} className="space-y-4">
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="startDate">Tanggal Mulai</Label>
                        <Input
                          id="startDate"
                          type="date"
                          value={bookingForm.startDate}
                          onChange={(e) => setBookingForm({...bookingForm, startDate: e.target.value})}
                          required
                        />
                      </div>
                      <div>
                        <Label htmlFor="endDate">Tanggal Selesai</Label>
                        <Input
                          id="endDate"
                          type="date"
                          value={bookingForm.endDate}
                          onChange={(e) => setBookingForm({...bookingForm, endDate: e.target.value})}
                          required
                        />
                      </div>
                    </div>

                    <div>
                      <Label htmlFor="pickupLocation">Lokasi Pengambilan</Label>
                      <Input
                        id="pickupLocation"
                        value={bookingForm.pickupLocation}
                        onChange={(e) => setBookingForm({...bookingForm, pickupLocation: e.target.value})}
                        placeholder="Masukkan lokasi pengambilan"
                        required
                      />
                    </div>

                    <div>
                      <Label htmlFor="name">Nama Lengkap</Label>
                      <Input
                        id="name"
                        value={bookingForm.name}
                        onChange={(e) => setBookingForm({...bookingForm, name: e.target.value})}
                        placeholder="Masukkan nama lengkap"
                        required
                      />
                    </div>

                    <div>
                      <Label htmlFor="email">Email</Label>
                      <Input
                        id="email"
                        type="email"
                        value={bookingForm.email}
                        onChange={(e) => setBookingForm({...bookingForm, email: e.target.value})}
                        placeholder="Masukkan email"
                        required
                      />
                    </div>

                    <div>
                      <Label htmlFor="phone">Nomor Telepon</Label>
                      <Input
                        id="phone"
                        value={bookingForm.phone}
                        onChange={(e) => setBookingForm({...bookingForm, phone: e.target.value})}
                        placeholder="Masukkan nomor telepon"
                        required
                      />
                    </div>

                    <div>
                      <Label htmlFor="notes">Catatan (Opsional)</Label>
                      <Textarea
                        id="notes"
                        value={bookingForm.notes}
                        onChange={(e) => setBookingForm({...bookingForm, notes: e.target.value})}
                        placeholder="Masukkan catatan tambahan"
                        rows={3}
                      />
                    </div>

                    {bookingForm.startDate && bookingForm.endDate && (
                      <div className="bg-gray-50 rounded-lg p-4">
                        <div className="flex justify-between items-center">
                          <span className="text-gray-600">Total Harga:</span>
                          <span className="text-2xl font-bold text-blue-600">
                            Rp {calculateTotalPrice().toLocaleString('id-ID')}
                          </span>
                        </div>
                      </div>
                    )}

                    <Button 
                      type="submit" 
                      className="w-full" 
                      disabled={!car.available || isBooking}
                    >
                      {isBooking ? 'Memproses...' : 'Pesan Sekarang'}
                    </Button>

                    {!car.available && (
                      <p className="text-center text-red-600 text-sm">
                        Mobil ini tidak tersedia untuk disewa
                      </p>
                    )}
                  </form>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  )
}